use std::fs::File;
use std::io::Write;
use std::time::SystemTime;

use anyhow;

use structopt::StructOpt;

use sequoia_openpgp as openpgp;
use openpgp::fmt::hex;
use openpgp::Result;
use openpgp::packet;
use openpgp::Packet;
use openpgp::packet::key::Key4;
use openpgp::packet::key::PublicParts;
use openpgp::packet::key::UnspecifiedRole;
use openpgp::serialize::SerializeInto;
use openpgp::serialize::stream::Armorer;
use openpgp::serialize::stream::Message;

use sequoia_tpm::Description;

#[derive(Debug, StructOpt)]
struct Opt {
    #[structopt(short, long)]
    file: String,
}

fn derive_key(pk: &sequoia_tpm::PublicKeyBytes)
    -> Result<packet::Key<PublicParts, UnspecifiedRole>>
{
    match pk {
        sequoia_tpm::PublicKeyBytes::RSA(pk) => {
            let bytes = hex::decode(&pk.bytes);
            match bytes {
                Ok(bytes) => {
                    Key4::import_public_rsa(
                        &[1, 0, 1], &bytes, SystemTime::UNIX_EPOCH)
                        .map(|k| packet::Key::V4(k))
                }
                Err(err) => {
                    Err(err.into())
                }
            }
        }
        _ => {
            Err(anyhow::anyhow!("Unsupported algorithm"))
        }
    }
}

fn main() -> Result<()> {
    env_logger::init();

    let opt = Opt::from_args();

    let filename = opt.file;

    let mut desc: Description = serde_yaml::from_reader(
        File::open(&filename)?)?;

    sequoia_tpm::read_key(&mut desc.spec)?;

    match sequoia_tpm::read_key(&mut desc.spec) {
        Ok(_key) => (),
        Err(err) => {
            eprintln!("Error configuring key ({}) on TPM: {}\n{}",
                      filename, err,
                      serde_yaml::to_string(&desc)?);
            return Err(err.into());
        }
    }

    let pk = if let Some(pk) = desc.spec.provider.tpm.unique.as_ref() {
        pk
    } else {
        eprintln!("{}: Unable to read public key.",
                  filename);
        return Err(anyhow::anyhow!("Unable to read public key"));
    };

    let key = match derive_key(pk) {
        Ok(key) => key,
        Err(err) => {
            eprintln!("{}: Failed to derive a null OpenPGP key: {}",
                      filename, err);
            return Err(err);
        }
    };

    let packet = Packet::from(
        key.role_into_primary());

    let packet = packet.to_vec()
        .expect("serializing to a vec is infallible");

    let mut stdout = std::io::stdout();

    let message = Message::new(&mut stdout);
    let mut message = Armorer::new(message)
        .kind(openpgp::armor::Kind::PublicKey)
        .build()?;
    message.write_all(&packet)?;
    message.finalize()?;

    Ok(())
}
