// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! Handle individual OpenPGP signatures.

pub mod stack;

use std::io;
use std::ops::Add;

use chrono::{DateTime, Utc};
use pgp::packet::{Packet, RevocationCode, SignatureType, SubpacketData, SubpacketType};
use pgp::ser::Serialize;
use pgp::{Deserializable, Signature, StandaloneSignature};

use crate::policy::acceptable_hash_algorithm;
use crate::Error;

fn is_revocation(sig: &Signature) -> bool {
    match sig.typ() {
        SignatureType::KeyRevocation
        | SignatureType::CertRevocation
        | SignatureType::SubkeyRevocation => {
            // this is a revocation

            true // unless explicitly marked as "soft" (by the reason code), we consider a revocation to be "hard"
        }
        _ => false, // this signature is not a revocation
    }
}

fn is_hard_revocation(sig: &Signature) -> bool {
    // FIXME: DRY with is_revocation
    match sig.typ() {
        SignatureType::KeyRevocation
        | SignatureType::CertRevocation
        | SignatureType::SubkeyRevocation => {
            // this is a revocation, but is it a hard revocation?

            match sig.revocation_reason_code() {
                Some(RevocationCode::KeyRetired)
                | Some(RevocationCode::CertUserIdInvalid)
                | Some(RevocationCode::KeySuperseded) => {
                    return false; // these are soft revocation codes
                }
                _ => {}
            }

            true // unless explicitly marked as "soft" (by the reason code), we consider a revocation to be "hard"
        }
        _ => false, // this signature is not a revocation
    }
}

/// Reports if a signature meets basic acceptability criteria, including the cryptographic
/// primitives it uses:
///
/// - Rejects signatures that have no creation time stamp set in the hashed area
/// - Rejects signatures that have a creation time stamp in the future
/// - Rejects signatures that contain unknown critical subpackets
///
/// md5 hashes are considered broken (effective January 1, 2010, based on the signature creation time).
/// We consider a more recently dated md5-based signature equally broken as one with an invalid cryptographic hash digest.
///
/// sha1 hashes for data signatures are considered broken effective January 1, 2014;
/// sha1 hashes for other signature types are considered broken effective February 1, 2023.
///
/// (Rejecting signatures that are technically correct, but use broken primitives is a defensive
/// tradeoff: we consider legacy signatures that were made after a cutoff time as either attacks or
/// mistakes.)
pub(crate) fn signature_acceptable(sig: &Signature) -> bool {
    // FIXME: break this fn up, it does too many different things

    let Some(sig_creation_time) = sig.created() else {
        return false;
    };

    // A signature with a future creation time is not currently valid
    let now: DateTime<Utc> = chrono::offset::Utc::now();
    if *sig_creation_time > now {
        return false;
    }

    // critical unknown subpackets or notations invalidate a signature
    for sp in &sig.config.hashed_subpackets {
        if sp.is_critical {
            if matches!(sp.typ(), SubpacketType::Other(_)) {
                // Unknown critical subpacket
                return false;
            }
            if let SubpacketData::Notation(_notation) = &sp.data {
                // Unknown critical notation (by default)
                // FIXME: how would an application use critical notations? initialize rpgpie with a good-list?
                return false;
            }
        }
    }

    let data_sig = sig.typ() == SignatureType::Binary || sig.typ() == SignatureType::Text;

    // reject signature if our policy rejects the hash algorithm at signature creation time
    if !acceptable_hash_algorithm(&sig.config.hash_alg, sig_creation_time, data_sig) {
        return false;
    }

    true
}

/// How long is `sig` valid?
///
/// Some(dt): valid until `dt`
/// None: unlimited validity
pub(crate) fn validity_end(sig: &Signature) -> Option<DateTime<Utc>> {
    let sig_creation = sig.created().expect("sig creation must be set");

    if let Some(sig_expiration) = sig.signature_expiration_time() {
        if sig_expiration.num_seconds() != 0 {
            Some(sig_creation.add(*sig_expiration))
        } else {
            None
        }
    } else {
        None
    }
}

pub(crate) fn is_signature_valid_at(
    sig: &Signature,
    key_creation: &DateTime<Utc>,
    reference: &DateTime<Utc>,
) -> bool {
    if let Some(creation) = sig.created() {
        // If the signature is created after the reference time, it is invalid at reference time
        if creation > reference {
            return false;
        }

        // If the signature expires before the reference time, the signature is invalid
        if let Some(sig_exp) = validity_end(sig) {
            if sig_exp < *reference {
                return false;
            }
        }

        // If the key expires and expiration is before the reference time, the signature is invalid
        if let Some(key_exp) = sig.key_expiration_time() {
            if key_exp.num_seconds() != 0 && (key_creation.add(*key_exp) < *reference) {
                return false;
            }
        }

        true
    } else {
        // A signature with unset creation time is invalid
        false
    }
}

/// Read a list of Signatures from an input
pub fn load<R: io::Read>(source: &mut R) -> Result<Vec<Signature>, Error> {
    let (iter, _header) = StandaloneSignature::from_reader_many(source)?;

    let mut sigs = vec![];

    for res in iter {
        if let Ok(sig) = res {
            sigs.push(sig.signature);
        } else {
            eprintln!("error reading signature {:?}", res);
        }
    }

    Ok(sigs)
}

/// Write a list of Signatures to an output
pub fn save(
    signatures: &[Signature],
    armored: bool,
    mut sink: &mut dyn io::Write,
) -> Result<(), Error> {
    if armored {
        let packets: Vec<_> = signatures.iter().map(|s| Packet::from(s.clone())).collect();

        pgp::armor::write(
            &packets,
            pgp::armor::BlockType::Signature,
            &mut sink,
            None,
            true,
        )?;
    } else {
        for s in signatures {
            let p = Packet::from(s.clone());
            p.to_writer(&mut sink)?;
        }
    }

    Ok(())
}
