// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! Wrapper types and logic around key material: Certificates and TSKs, component keys.

use std::io;

use chrono::{DateTime, Utc};
use openpgp_card_rpgp::CardSlot;
use pgp::cleartext::CleartextSignedMessage;
use pgp::crypto::hash::HashAlgorithm;
use pgp::ser::Serialize;
use pgp::types::{KeyTrait, SignedUser};
use pgp::{
    ArmorOptions, KeyType, Message, PublicOrSecret, SecretKeyParamsBuilder, Signature,
    SignedKeyDetails, SignedPublicKey, SignedSecretKey, StandaloneSignature, SubkeyParamsBuilder,
};

use crate::card::{card_by_pp, verify_pin_from_card_state};
use crate::key::component::{
    ComponentKeyPub, ComponentKeySec, KeyFlagMatch, SignedComponentKey, SignedComponentKeyPub,
    SignedComponentKeySec,
};
use crate::policy::{
    PREFERRED_COMPRESSION_ALGORITHMS, PREFERRED_HASH_ALGORITHMS, PREFERRED_SYMMETRIC_KEY_ALGORITHMS,
};
use crate::sig::stack::SigStack;
use crate::{card, policy, Error};

pub mod checked;
pub mod component;

/// A "certificate," also known as an "OpenPGP public key."
#[derive(Debug, Clone)]
pub struct Certificate {
    spk: SignedPublicKey,
}

impl From<SignedPublicKey> for Certificate {
    fn from(spk: SignedPublicKey) -> Self {
        Self { spk }
    }
}

impl TryFrom<&[u8]> for Certificate {
    type Error = Error;

    fn try_from(input: &[u8]) -> Result<Self, Self::Error> {
        use pgp::Deserializable;

        let (spk, _) = SignedPublicKey::from_reader_single(input)?;
        Ok(Self { spk })
    }
}

impl TryFrom<&Certificate> for Vec<u8> {
    type Error = Error;

    fn try_from(value: &Certificate) -> Result<Self, Self::Error> {
        Ok(value.spk().to_bytes()?)
    }
}

impl From<&Tsk> for Certificate {
    fn from(value: &Tsk) -> Self {
        match value {
            Tsk::Card(c) => c.clone(),
            Tsk::Tsk(spk) => Certificate::from(SignedPublicKey::from(spk.clone())),
        }
    }
}

/// pick the user id we consider primary
fn pick_primary_user_id<'a>(users: &'_ [&'a SignedUser]) -> Option<&'a SignedUser> {
    match users.len() {
        0 => None,
        1 => Some(users[0]),

        _ => {
            // FIXME: pick the user id with the most recent self-signature?
            // (ignoring revoked user ids, unless all are revoked)

            let mut best = users[0];

            let now = Utc::now(); // FIXME?

            let stack = SigStack::from_iter(&best.signatures);
            let mut best_is_revoked = stack.revoked_at(&now);
            // let mut best_latest_sig = stack.active_at(Some(&now));

            for user in users {
                let stack = SigStack::from_iter(&user.signatures);
                let this_is_revoked = stack.revoked_at(&now);
                // let this_latest_sig = stack.active_at(Some(&now));

                if best_is_revoked && !this_is_revoked {
                    // replace
                    best = user;
                    best_is_revoked = this_is_revoked;
                    // best_latest_sig = this_latest_sig;
                }
            }

            Some(best)
        }
    }
}

/// Return primary User ID:
///
/// We prefer the User ID with the most recent self-signature.
///
/// Either from among all User IDs that are marked "primary".
/// Or from among all User IDs if none are marked "primary".
///
/// (Note: we don't implement looking up the primary User ID at a reference time!)
fn primary_user_id(details: &SignedKeyDetails) -> Option<&SignedUser> {
    // Find all User IDs that are marked "primary"
    let primaries: Vec<_> = details.users.iter().filter(|u| u.is_primary()).collect();

    if !primaries.is_empty() {
        // choose between all that are marked "primary"
        pick_primary_user_id(&primaries)
    } else {
        // No User ID was marked "primary", we choose among all User IDs.
        let users: Vec<_> = details.users.iter().collect();
        pick_primary_user_id(&users)
    }
}

/// Returns the latest signature of the primary User ID.
/// None, if no primary User ID or binding signature for it are found.
pub(crate) fn primary_user_id_binding_at<'a>(
    details: &'a SignedKeyDetails,
    reference: &'a DateTime<Utc>,
) -> Option<&'a Signature> {
    primary_user_id(details)
        .and_then(|uid| SigStack::from_iter(uid.signatures.iter()).active_at(Some(reference)))
}

impl Certificate {
    pub fn fingerprint(&self) -> Vec<u8> {
        self.spk.primary_key.fingerprint()
    }

    pub fn load<R: io::Read>(source: &mut R) -> Result<Vec<Certificate>, Error> {
        let mut certs = vec![];

        let (parsed, _headers) = pgp::composed::signed_key::from_reader_many(source)?;

        for res in parsed {
            match res {
                Ok(pos) => {
                    let cert = match pos {
                        PublicOrSecret::Public(spk) => spk.into(),
                        PublicOrSecret::Secret(_ssk) => {
                            return Err(Error::Message(
                                "Expected Certificate(s), got TSK".to_string(),
                            ));
                        }
                    };

                    certs.push(cert);
                }
                Err(_) => eprintln!("Bad data {:?}", res),
            }
        }

        if certs.is_empty() {
            Err(Error::Message("No certificates found".to_string()))
        } else {
            Ok(certs)
        }
    }

    pub fn save(
        certs: &Vec<Self>,
        armored: bool,
        mut sink: &mut dyn io::Write,
    ) -> Result<(), Error> {
        if armored {
            if certs.len() != 1 {
                // FIXME: handle cert-rings
                return Err(Error::Message(
                    "can currently only save exactly one Certificate".to_string(),
                ));
            }

            let spk = &certs[0].spk();

            spk.to_armored_writer(&mut sink, ArmorOptions::default())?;
        } else {
            for c in certs {
                c.spk().to_writer(&mut sink)?;
            }
        }

        Ok(())
    }

    pub(crate) fn spk(&self) -> &SignedPublicKey {
        &self.spk
    }
}

/// A "transferable secret key (TSK)," also known as an "OpenPGP private/secret key."
#[derive(Clone)]
pub enum Tsk {
    Tsk(SignedSecretKey),
    Card(Certificate),
}

impl TryFrom<&[u8]> for Tsk {
    type Error = Error;

    fn try_from(input: &[u8]) -> Result<Self, Self::Error> {
        use pgp::Deserializable;

        let (ssk, _) = SignedSecretKey::from_reader_single(input)?;
        Ok(Self::Tsk(ssk))
    }
}

impl From<SignedSecretKey> for Tsk {
    fn from(tsk: SignedSecretKey) -> Self {
        Tsk::Tsk(tsk)
    }
}

pub struct SignatureVerifier {
    ckey: ComponentKeyPub,
}

impl SignatureVerifier {
    pub fn verify(&self, signature: &Signature, data: &[u8]) -> Result<(), Error> {
        self.ckey.verify(signature, data)
    }

    pub fn verify_csf(&self, csf: &CleartextSignedMessage) -> Result<StandaloneSignature, Error> {
        match &self.ckey {
            ComponentKeyPub::Primary(pk) => Ok(csf.verify(pk)?),
            ComponentKeyPub::Subkey(psk) => Ok(csf.verify(psk)?),
        }
        .cloned()
    }

    pub fn as_componentkey(&self) -> &ComponentKeyPub {
        &self.ckey
    }
}

impl From<ComponentKeyPub> for SignatureVerifier {
    fn from(ckey: ComponentKeyPub) -> Self {
        SignatureVerifier { ckey }
    }
}

impl From<SignatureVerifier> for ComponentKeyPub {
    fn from(value: SignatureVerifier) -> Self {
        value.ckey
    }
}

pub enum DataSigner {
    Software(ComponentKeySec),
    Card(ComponentKeyPub),
}

impl DataSigner {
    pub fn sign_msg<F>(
        &self,
        msg: Message,
        key_pw: F,
        hash_algo: HashAlgorithm,
    ) -> Result<Message, Error>
    where
        F: FnOnce() -> String,
    {
        match self {
            DataSigner::Software(s) => Ok(s.sign_msg(msg, key_pw, hash_algo)?),
            DataSigner::Card(ckp) => {
                // FIXME: upstream public_params() call
                let pp = match ckp {
                    ComponentKeyPub::Primary(p) => p.public_params(),
                    ComponentKeyPub::Subkey(s) => s.public_params(),
                };

                // FIXME: allow users to pass in a touch prompt?
                Ok(card::sign_on_card(&msg, pp, hash_algo, &|| {})?)
            }
        }
    }

    pub fn sign_csf(&self, body: &str, key_pws: &[&[u8]]) -> Result<CleartextSignedMessage, Error> {
        // FIXME: try using `key_pw` at least for softkeys

        match self {
            DataSigner::Card(p) => {
                if let Some(mut card) =
                    card_by_pp(p.public_params(), openpgp_card::ocard::KeyType::Signing)?
                {
                    let mut tx = card.transaction()?;
                    verify_pin_from_card_state(tx.card(), true)?;

                    // FIXME: allow users to pass in a touch prompt?
                    let cs = CardSlot::init_from_card(
                        &mut tx,
                        openpgp_card::ocard::KeyType::Signing,
                        &|| {},
                    )?;

                    Ok(CleartextSignedMessage::sign(body, &cs, String::default)?)
                } else {
                    Err(Error::Message("Card not found".to_string()))
                }
            }
            DataSigner::Software(cks) => match cks {
                // FIXME: DRY
                ComponentKeySec::Primary(sk) => key_pws
                    .iter()
                    .flat_map(|key_pw| {
                        CleartextSignedMessage::sign(body, &sk, || {
                            String::from_utf8_lossy(key_pw).to_string()
                        })
                    })
                    .next(),
                ComponentKeySec::Subkey(ssk) => key_pws
                    .iter()
                    .flat_map(|key_pw| {
                        CleartextSignedMessage::sign(body, &ssk, || {
                            String::from_utf8_lossy(key_pw).to_string()
                        })
                    })
                    .next(),
            }
            .ok_or(Error::Message(
                "Couldn't sign. No suitable password found?".to_string(),
            )),
        }
    }

    pub fn fingerprint(&self) -> Vec<u8> {
        match self {
            DataSigner::Card(p) => p.fingerprint(),
            DataSigner::Software(s) => match s {
                ComponentKeySec::Primary(sk) => sk.fingerprint(),
                ComponentKeySec::Subkey(ssk) => ssk.fingerprint(),
            },
        }
    }
}

impl Tsk {
    pub fn load<R: io::Read>(read: &mut R) -> Result<Vec<Tsk>, Error> {
        let (keys, _headers) = pgp::composed::signed_key::from_reader_many(read)?;

        let mut tsk = vec![];

        for res in keys {
            match res {
                Ok(pos) => match pos {
                    PublicOrSecret::Public(spk) => {
                        let card = Tsk::Card(spk.into());
                        tsk.push(card);
                    }
                    PublicOrSecret::Secret(ssk) => {
                        tsk.push(ssk.into());
                    }
                },
                Err(_) => eprintln!("Bad data {:?}", res),
            }
        }

        if tsk.is_empty() {
            Err(Error::Message("No TSKs found".to_string()))
        } else {
            Ok(tsk)
        }
    }

    pub fn save(tsks: &[Self], armored: bool, mut sink: &mut dyn io::Write) -> Result<(), Error> {
        if tsks.len() != 1 {
            // FIXME: handle keyrings
            return Err(Error::Message(
                "Can currently only save exactly one TSK".to_string(),
            ));
        }

        let signed_secret_key: &Tsk = &tsks[0];

        if armored {
            signed_secret_key
                .key()
                .to_armored_writer(&mut sink, ArmorOptions::default())?;
        } else {
            signed_secret_key.key().to_writer(&mut sink)?;
        }

        Ok(())
    }

    // FIXME: remove this from the API again?
    pub fn key(&self) -> &SignedSecretKey {
        match self {
            Tsk::Tsk(ssk) => ssk,
            Tsk::Card(_) => panic!("card mode Tsk, only usable for HSM operations"), // FIXME
        }
    }

    pub fn generate(
        key_type_primary: KeyType,
        key_type_encrypt: impl Into<Option<KeyType>>,
        primary_user_id: Option<String>,
        secondary_user_ids: Vec<String>,
    ) -> Result<Self, Error> {
        let Some(primary_user_id) = primary_user_id else {
            panic!("Generating UID-less keys not supported");
        };

        let subkeys = if let Some(key_type_encrypt) = key_type_encrypt.into() {
            vec![SubkeyParamsBuilder::default()
                .key_type(key_type_encrypt)
                .can_encrypt(true)
                .build()?]
        } else {
            vec![]
        };

        let mut key_params = SecretKeyParamsBuilder::default();
        key_params
            .key_type(key_type_primary)
            .can_certify(true)
            .can_sign(true)
            .primary_user_id(primary_user_id)
            .user_ids(secondary_user_ids)
            .preferred_symmetric_algorithms(PREFERRED_SYMMETRIC_KEY_ALGORITHMS.into())
            .preferred_hash_algorithms(PREFERRED_HASH_ALGORITHMS.into())
            .preferred_compression_algorithms(PREFERRED_COMPRESSION_ALGORITHMS.into())
            .subkeys(subkeys);

        let secret_key_params = key_params.build()?;
        let secret_key = secret_key_params.generate()?;

        let passwd_fn = String::new;
        let signed_secret_key = secret_key.sign(passwd_fn)?;

        Ok(Tsk::Tsk(signed_secret_key))
    }

    fn component_keys(&self) -> Vec<SignedComponentKey> {
        match self {
            Tsk::Tsk(tsk) => {
                let x = SignedComponentKeySec::Primary(tsk.clone());
                let mut v = vec![SignedComponentKey::Sec(x)];

                tsk.secret_subkeys.iter().for_each(|sssk| {
                    let dks = SigStack::from_iter(tsk.details.direct_signatures.iter()).active();

                    let x = SignedComponentKeySec::Subkey((sssk.clone(), dks.cloned()));
                    v.push(SignedComponentKey::Sec(x));
                });

                v
            }
            Tsk::Card(cert) => {
                let spk = &cert.spk;

                // FIXME: collecting a copy of all sigs is weird
                let mut sigs: Vec<_> = spk.details.revocation_signatures.clone();
                sigs.append(&mut spk.details.direct_signatures.clone());

                let x = SignedComponentKeyPub::Primary((spk.clone(), sigs));
                let mut v = vec![SignedComponentKey::Pub(x)];

                spk.public_subkeys.iter().for_each(|sps| {
                    let dks = SigStack::from_iter(spk.details.direct_signatures.iter()).active();

                    let x = SignedComponentKeyPub::Subkey((sps.clone(), dks.cloned()));
                    v.push(SignedComponentKey::Pub(x));
                });

                v
            }
        }
    }

    /// Get list of all signing capable component keys
    /// (this fn is specific to *signing*, not validation: it uses "now" as its reference time and is thus stricter)
    pub fn signing_capable_component_keys(&self) -> impl Iterator<Item = DataSigner> + '_ {
        let now: DateTime<Utc> = chrono::offset::Utc::now();

        // FIXME: this should probably also do the checks that `CorrectCertificate` does

        let (pri_pp, pri_details) = match self {
            Tsk::Tsk(ssk) => (ssk.primary_key.public_params(), &ssk.details),
            Tsk::Card(c) => (c.spk.primary_key.public_params(), &c.spk.details),
        };

        // If the primary is too weak, we must reject all of its components
        let primary_acceptable = policy::acceptable_pk_algorithm(pri_pp, &now)
            && !SigStack::from_iter(pri_details.revocation_signatures.iter()).revoked_at(&now);

        self.component_keys()
            .into_iter()
            .filter(move |_| primary_acceptable)
            .filter(move |skc| policy::acceptable_pk_algorithm(skc.public_params(), &now))
            .filter(move |sck| sck.is_signing_capable(&now) && !sck.revoked(&now))
            .map(|x| match x {
                SignedComponentKey::Pub(p) => DataSigner::Card(p.into()),
                SignedComponentKey::Sec(p) => DataSigner::Software((&p).into()),
            })
    }

    /// Get list of all decryption capable component keys
    /// (this fn is specific to *decryption*, not encryption: it is very lenient in allowing use of keys)
    pub fn decryption_capable_component_keys(
        &self,
    ) -> impl Iterator<Item = SignedComponentKey> + '_ {
        let now: DateTime<Utc> = chrono::offset::Utc::now();

        // FIXME: filter out unknown notations

        self.component_keys()
            .into_iter()
            .filter(move |sck| sck.is_encryption_capable(&now))
    }

    fn sec_components(&self) -> Vec<SignedComponentKeySec> {
        match self {
            Tsk::Tsk(ssk) => {
                let x = SignedComponentKeySec::Primary(ssk.clone());
                let mut v = vec![x];

                ssk.secret_subkeys.iter().for_each(|sssk| {
                    let dks = SigStack::from_iter(ssk.details.direct_signatures.iter()).active();

                    let x = SignedComponentKeySec::Subkey((sssk.clone(), dks.cloned()));
                    v.push(x);
                });

                v
            }
            Tsk::Card(_) => vec![],
        }
    }

    /// Get all component secret keys that have a "signing capable" key flag.
    ///
    /// NOTE: this function doesn't cryptographically verify self-signatures, it's not safe to use
    /// on potentially attacker-controlled Tsks.
    pub fn signing_keys_sec(&self) -> Vec<ComponentKeySec> {
        // FIXME: filter out revoked subkeys

        self.sec_components()
            .iter()
            .filter(|x| x.has_key_flag(KeyFlagMatch::Sign))
            .map(Into::into)
            .collect()
    }

    /// Get all component secret keys that have an "encryption capable" key flag.
    ///
    /// NOTE: this function doesn't cryptographically verify self-signatures, it's not safe to use
    /// on potentially attacker-controlled Tsks.
    pub fn decryption_keys_sec(&self) -> Vec<ComponentKeySec> {
        // FIXME: filter out revoked subkeys

        self.sec_components()
            .iter()
            .filter(|x| x.has_key_flag(KeyFlagMatch::Enc))
            .map(Into::into)
            .collect()
    }

    /// Get all component secret keys that have an "authentication capable" key flag.
    ///
    /// NOTE: this function doesn't cryptographically verify self-signatures, it's not safe to use
    /// on potentially attacker-controlled Tsks.
    pub fn auth_keys_sec(&self) -> Vec<ComponentKeySec> {
        // FIXME: filter out revoked subkeys

        self.sec_components()
            .iter()
            .filter(|x| x.has_key_flag(KeyFlagMatch::Auth))
            .map(Into::into)
            .collect()
    }
}
