// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! Handling of component keys. A component key is an individual "primary key" or "subkey."
//!
//! NOTE: This module is particularly experimental, the API may change drastically.
//! (This current implementation does a lot of cloning and is limited to read-only operations.)

use chrono::{DateTime, Utc};
use pgp::crypto::hash::HashAlgorithm;
use pgp::crypto::public_key::PublicKeyAlgorithm;
use pgp::crypto::sym::SymmetricKeyAlgorithm;
use pgp::decrypt_session_key;
use pgp::packet::{KeyFlags, PublicKeyEncryptedSessionKey, SubpacketData};
use pgp::types::{KeyId, KeyTrait, PublicParams};
use pgp::{
    packet, Message, Signature, SignedPublicKey, SignedPublicSubKey, SignedSecretKey,
    SignedSecretSubKey,
};
use rand::Rng;
use rand_core::CryptoRng;

use crate::key::primary_user_id_binding_at;
use crate::sig::stack::SigStack;
use crate::{key, sig, util, Error};

/// Any kind of component key: public or secret, primary or subkey.
/// This enum serves as a convenience wrapper around all four cases.
///
/// The component key is stored combined with the context of its binding signature(s)
pub enum SignedComponentKey {
    Pub(SignedComponentKeyPub),
    Sec(SignedComponentKeySec),
}

enum SignedComponentPrimary {
    Pub(SignedPublicKey),
    Sec(SignedSecretKey),
}

impl SignedComponentPrimary {
    fn direct_key_signature_at(&self, reference: &DateTime<Utc>) -> Option<&Signature> {
        match self {
            SignedComponentPrimary::Pub(spk) => {
                SigStack::from_iter(spk.details.direct_signatures.iter()).active_at(Some(reference))
            }
            SignedComponentPrimary::Sec(ssk) => {
                SigStack::from_iter(ssk.details.direct_signatures.iter()).active_at(Some(reference))
            }
        }
    }

    fn primary_user_id_binding_at<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> Option<&'a Signature> {
        match self {
            SignedComponentPrimary::Pub(p) => {
                key::primary_user_id_binding_at(&p.details, reference)
            }
            SignedComponentPrimary::Sec(s) => {
                key::primary_user_id_binding_at(&s.details, reference)
            }
        }
    }
}

impl SignedComponentSubkey {
    // FIXME: should probably also keep track of primary user id binding signature
    fn direct_key_signature(&self) -> Option<&Signature> {
        match self {
            SignedComponentSubkey::Pub((_, dks)) => dks.as_ref(),
            SignedComponentSubkey::Sec((_, dks)) => dks.as_ref(),
        }
    }

    fn subkey_binding_at(&self, reference: Option<&DateTime<Utc>>) -> Option<&Signature> {
        match self {
            SignedComponentSubkey::Pub((spsk, _)) => {
                SigStack::from_iter(spsk.signatures.iter()).active_at(reference)
            }
            SignedComponentSubkey::Sec((sssk, _)) => {
                SigStack::from_iter(sssk.signatures.iter()).active_at(reference)
            }
        }
    }
}

#[derive(Debug, Clone)]
enum SignedComponentSubkey {
    Pub((SignedPublicSubKey, Option<Signature>)), // key, dks
    Sec((SignedSecretSubKey, Option<Signature>)), // key, dks
}

/// Signed component key, either a primary or a subkey
enum PrimaryOrSubkey {
    Primary(SignedComponentPrimary),
    Subkey(SignedComponentSubkey),
}

impl From<&SignedComponentKey> for PrimaryOrSubkey {
    fn from(value: &SignedComponentKey) -> Self {
        // FIXME: don't clone
        match value {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((k, _))) => {
                PrimaryOrSubkey::Primary(SignedComponentPrimary::Pub(k.clone()))
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey(pair)) => {
                PrimaryOrSubkey::Subkey(SignedComponentSubkey::Pub(pair.clone()))
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(k)) => {
                PrimaryOrSubkey::Primary(SignedComponentPrimary::Sec(k.clone()))
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey(pair)) => {
                PrimaryOrSubkey::Subkey(SignedComponentSubkey::Sec(pair.clone()))
            }
        }
    }
}

impl SignedComponentKey {
    pub fn version(&self) -> u8 {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((spk, _))) => {
                spk.primary_key.version().into()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                spsk.key.version().into()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(ssk)) => {
                ssk.primary_key.version().into()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                sssk.key.version().into()
            }
        }
    }

    pub fn public_params(&self) -> &PublicParams {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((spk, _))) => {
                spk.primary_key.public_params()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                spsk.key.public_params()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(ssk)) => {
                ssk.primary_key.public_params()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                sssk.key.public_params()
            }
        }
    }

    pub fn fingerprint(&self) -> Vec<u8> {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((spk, _))) => {
                spk.primary_key.fingerprint()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                spsk.key.fingerprint()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(ssk)) => {
                ssk.primary_key.fingerprint()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                sssk.key.fingerprint()
            }
        }
    }

    pub fn key_id(&self) -> KeyId {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((spk, _))) => {
                spk.primary_key.key_id()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => spsk.key.key_id(),
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(ssk)) => {
                ssk.primary_key.key_id()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => sssk.key.key_id(),
        }
    }

    pub fn created_at(&self) -> &DateTime<Utc> {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((spk, _))) => {
                spk.primary_key.created_at()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                spsk.key.created_at()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(ssk)) => {
                ssk.primary_key.created_at()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                sssk.key.created_at()
            }
        }
    }

    pub fn sigs(&self) -> SigStack {
        match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((_spk, sigs))) => {
                SigStack::from_iter(sigs.iter())
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                SigStack::from_iter(spsk.signatures.iter())
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(_ssk)) => {
                todo!() // FIXME
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                SigStack::from_iter(sssk.signatures.iter())
            }
        }
    }

    /// Find KeyFlags subpacket in hashed area, return first occurrence (if any)
    fn key_flag_subpacket(sig: &Signature) -> Option<KeyFlags> {
        sig.config
            .hashed_subpackets
            .iter()
            .find_map(|p| match &p.data {
                SubpacketData::KeyFlags(d) => Some(d[..].into()),
                _ => None,
            })
    }

    /// Get key flags.
    ///
    /// This convenience fn takes into consideration the different places that key flags can be stored in:
    /// - For primary component keys: dks or primary user id binding
    /// - For component subkeys: subkey binding or dks
    pub(crate) fn key_flags_at(&self, reference: &DateTime<Utc>) -> Option<KeyFlags> {
        match PrimaryOrSubkey::from(self) {
            PrimaryOrSubkey::Primary(scp) => {
                // We favor the dsk for the primary
                if let Some(dks) = &scp.direct_key_signature_at(reference) {
                    if let Some(kf) = Self::key_flag_subpacket(dks) {
                        return Some(kf);
                    }
                } else if let Some(prim_bind) = scp.primary_user_id_binding_at(reference) {
                    if let Some(kf) = Self::key_flag_subpacket(prim_bind) {
                        return Some(kf);
                    }
                }

                Some(KeyFlags::default())
            }

            PrimaryOrSubkey::Subkey(scsk) => {
                let Some(binding) = scsk.subkey_binding_at(Some(reference)) else {
                    eprintln!("no subkey binding");
                    return None;
                };

                // We favor the subkey binding for subkeys
                if let Some(kf) = Self::key_flag_subpacket(binding) {
                    return Some(kf);
                } else if let Some(dks) = scsk.direct_key_signature() {
                    // FIXME: should we also consider the primary user id binding here?
                    // (or should we just find the primary key's flags and use those?!)

                    if let Some(kf) = Self::key_flag_subpacket(dks) {
                        return Some(kf);
                    }
                }

                Some(KeyFlags::default())
            }
        }
    }

    /// Check if *this* component key is revoked at `reference`.
    /// (Note that for subkeys, the revocation status of the primary is not considered!)
    pub fn revoked(&self, reference: &DateTime<Utc>) -> bool {
        let sigs = match self {
            SignedComponentKey::Pub(SignedComponentKeyPub::Primary((k, _))) => {
                k.details.revocation_signatures.as_slice()
            }
            SignedComponentKey::Pub(SignedComponentKeyPub::Subkey((spsk, _))) => {
                spsk.signatures.as_slice()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Primary(k)) => {
                k.details.revocation_signatures.as_slice()
            }
            SignedComponentKey::Sec(SignedComponentKeySec::Subkey((sssk, _))) => {
                sssk.signatures.as_slice()
            }
        };

        SigStack::from_iter(sigs.iter()).revoked_at(reference)
    }

    pub(crate) fn is_encryption_capable(&self, reference: &DateTime<Utc>) -> bool {
        let Some(kf) = self.key_flags_at(reference) else {
            return false;
        };

        kf.encrypt_comms() || kf.encrypt_storage()
    }

    pub(crate) fn is_signing_capable(&self, reference: &DateTime<Utc>) -> bool {
        let Some(kf) = self.key_flags_at(reference) else {
            return false;
        };

        kf.sign()
    }

    pub(crate) fn is_authentication_capable(&self, reference: &DateTime<Utc>) -> bool {
        let Some(kf) = self.key_flags_at(reference) else {
            return false;
        };

        kf.authentication()
    }
}

/// Either a public primary key, or a public subkey.
/// This enum serves as a convenience wrapper around both cases.
///
/// The component key consists of only the raw key packet, without the context of binding signatures

#[derive(Debug, Clone)]
pub enum ComponentKeyPub {
    Primary(packet::PublicKey),
    Subkey(packet::PublicSubkey),
}

#[derive(Debug, Clone)]
pub enum SignedComponentKeyPub {
    // We store a copy of all primary-related signatures, to have an owned copy that can be borrowed as a SigStack.
    // FIXME: do this better?
    Primary((SignedPublicKey, Vec<Signature>)),

    // key, dks
    Subkey((SignedPublicSubKey, Option<Signature>)),
}

impl From<SignedComponentKeyPub> for ComponentKeyPub {
    fn from(value: SignedComponentKeyPub) -> ComponentKeyPub {
        match value {
            SignedComponentKeyPub::Subkey(sk) => ComponentKeyPub::Subkey(sk.0.key),
            SignedComponentKeyPub::Primary((pk, _)) => {
                ComponentKeyPub::Primary(pk.primary_key.clone())
            }
        }
    }
}

#[allow(dead_code)]
pub enum ComponentKeySec {
    Primary(packet::SecretKey),
    Subkey(packet::SecretSubkey),
}

impl ComponentKeySec {
    pub fn sign_msg<F>(
        &self,
        msg: Message,
        key_pw: F,
        hash_algo: HashAlgorithm,
    ) -> Result<Message, Error>
    where
        F: FnOnce() -> String,
    {
        match self {
            ComponentKeySec::Primary(sk) => Ok(msg.sign(&sk, key_pw, hash_algo)?),
            ComponentKeySec::Subkey(ssk) => Ok(msg.sign(&ssk, key_pw, hash_algo)?),
        }
    }

    pub fn key_id(&self) -> KeyId {
        match self {
            ComponentKeySec::Primary(pri) => pri.key_id(),
            ComponentKeySec::Subkey(sub) => sub.key_id(),
        }
    }

    pub fn fingerprint(&self) -> Vec<u8> {
        match self {
            ComponentKeySec::Primary(pri) => pri.fingerprint(),
            ComponentKeySec::Subkey(sub) => sub.fingerprint(),
        }
    }

    pub fn created_at(&self) -> &DateTime<Utc> {
        match self {
            ComponentKeySec::Primary(pri) => pri.created_at(),
            ComponentKeySec::Subkey(sub) => sub.created_at(),
        }
    }

    pub fn algorithm(&self) -> PublicKeyAlgorithm {
        match self {
            ComponentKeySec::Primary(pri) => pri.algorithm(),
            ComponentKeySec::Subkey(sub) => sub.algorithm(),
        }
    }

    pub fn public_params(&self) -> &PublicParams {
        match self {
            ComponentKeySec::Primary(pri) => pri.public_params(),
            ComponentKeySec::Subkey(sub) => sub.public_params(),
        }
    }
}

/// Either a secret primary key, or a secret subkey.
/// This enum serves as a convenience wrapper around both cases.
///
/// The component key is stored combined with the context of its binding signature(s)
pub enum SignedComponentKeySec {
    Primary(SignedSecretKey),
    Subkey((SignedSecretSubKey, Option<Signature>)), // key, dks
}

pub(crate) enum KeyFlagMatch {
    Sign,
    Enc,
    Auth,
}

impl SignedComponentKeySec {
    pub fn key_id(&self) -> KeyId {
        match self {
            SignedComponentKeySec::Primary(ssk) => ssk.key_id(),
            SignedComponentKeySec::Subkey((sssk, _)) => sssk.key_id(),
        }
    }

    fn match_flag(sig: &Signature, check: KeyFlagMatch) -> bool {
        let flags = sig.key_flags();
        match check {
            KeyFlagMatch::Sign => flags.sign(),
            KeyFlagMatch::Auth => flags.authentication(),
            KeyFlagMatch::Enc => flags.encrypt_comms() | flags.encrypt_storage(),
        }
    }

    pub(crate) fn has_key_flag(&self, check: KeyFlagMatch) -> bool {
        match self {
            SignedComponentKeySec::Primary(ssk) => {
                let x = &ssk.details;

                // FIXME: also consider direct signatures
                if let Some(binding) = primary_user_id_binding_at(x, &Utc::now()) {
                    Self::match_flag(binding, check)
                } else {
                    false
                }
            }
            SignedComponentKeySec::Subkey((sssk, _)) => {
                let stack = SigStack::from_iter(sssk.signatures.iter());
                if let Some(sig) = stack.active_at(Some(&Utc::now())) {
                    Self::match_flag(sig, check)
                } else {
                    false
                }
            }
        }
    }

    pub fn decrypt_session_key<F>(
        &self,
        pkesk: &PublicKeyEncryptedSessionKey,
        key_pw: F,
    ) -> Result<pgp::PlainSessionKey, Error>
    where
        F: FnOnce() -> String,
    {
        match self {
            SignedComponentKeySec::Primary(ssk) => {
                Ok(decrypt_session_key(&ssk, key_pw, pkesk.mpis())?)
            }
            SignedComponentKeySec::Subkey((sssk, _)) => {
                Ok(decrypt_session_key(&sssk, key_pw, pkesk.mpis())?)
            }
        }
    }
}

impl From<&SignedComponentKeySec> for ComponentKeySec {
    fn from(value: &SignedComponentKeySec) -> Self {
        match value {
            SignedComponentKeySec::Primary(ssk) => {
                ComponentKeySec::Primary(ssk.primary_key.clone())
            }
            SignedComponentKeySec::Subkey((sssk, _)) => ComponentKeySec::Subkey(sssk.key.clone()),
        }
    }
}

impl SignedComponentKeyPub {
    pub fn algorithm_name(&self) -> String {
        match &self {
            SignedComponentKeyPub::Primary((spk, _)) => {
                util::algo_name(spk.primary_key.public_params())
            }
            SignedComponentKeyPub::Subkey((spsk, _)) => util::algo_name(spsk.key.public_params()),
        }
    }

    pub(crate) fn key_creation_time(&self) -> &DateTime<Utc> {
        match &self {
            SignedComponentKeyPub::Primary((spk, _)) => spk.primary_key.created_at(),
            SignedComponentKeyPub::Subkey((spsk, _)) => spsk.key.created_at(),
        }
    }

    /// This fn is intended for use on signing-capable component keys.
    /// It returns "false" is a subkey is not validly "backward-bound" to the primary.
    pub(crate) fn has_valid_backsig_at(
        &self,
        key_creation: &DateTime<Utc>,
        reference: &DateTime<Utc>,
    ) -> bool {
        if let SignedComponentKeyPub::Subkey(sssk) = self {
            // If a subkey binding signature has no embedded back signature,
            // the subkey is not reasonably bound to the certificate for signing

            let binding = SigStack::from_iter(sssk.0.signatures.iter()).active_at(Some(reference));

            if let Some(binding) = binding {
                binding
                    .embedded_signature()
                    .iter()
                    .any(|backsig| sig::is_signature_valid_at(backsig, key_creation, reference))
            } else {
                // we don't even have a binding at the reference time
                false
            }
        } else {
            // A primary doesn't need a backsig to be "validly bound"

            true
        }
    }

    /// This fn presupposes that the primary is valid at the reference time
    pub(crate) fn is_component_subkey_valid_at(&self, reference: &DateTime<Utc>) -> bool {
        match &self {
            SignedComponentKeyPub::Subkey(sk) => Self::is_subkey_valid_at(&sk.0, reference),
            SignedComponentKeyPub::Primary(_) => true,
        }
    }

    /// takes into account the semantics of hard and soft revocation
    fn subkey_revoked_at(sk: &SignedPublicSubKey, reference: &DateTime<Utc>) -> bool {
        SigStack::from_iter(sk.signatures.iter()).revoked_at(reference)
    }

    fn is_subkey_valid_at(spsk: &SignedPublicSubKey, reference: &DateTime<Utc>) -> bool {
        // subkey is revoked
        if Self::subkey_revoked_at(spsk, reference) {
            return false;
        }
        let key_creation = spsk.key.created_at();

        let stack = SigStack::from_iter(spsk.signatures.iter());
        if !stack.has_valid_binding_at(reference, key_creation) {
            return false;
        }

        true
    }

    /// Are we willing to encrypt to the algorithm/ECDH parameters of self?
    pub(crate) fn valid_encryption_algo(&self) -> bool {
        // FIXME: where should this go?

        let pp = self.public_params();

        crate::policy::accept_for_encryption(pp)
    }

    pub(crate) fn public_params(&self) -> &PublicParams {
        match &self {
            SignedComponentKeyPub::Primary((pk, _)) => pk.primary_key.public_params(),
            SignedComponentKeyPub::Subkey((sk, _)) => sk.key.public_params(),
        }
    }
}

impl ComponentKeyPub {
    pub fn pkesk_from_session_key<R: CryptoRng + Rng>(
        &self,
        rng: &mut R,
        session_key: &[u8],
        alg: SymmetricKeyAlgorithm,
    ) -> Result<PublicKeyEncryptedSessionKey, Error> {
        match &self {
            Self::Primary(pk) => Ok(PublicKeyEncryptedSessionKey::from_session_key(
                rng,
                session_key,
                alg,
                pk,
            )?),
            Self::Subkey(psk) => Ok(PublicKeyEncryptedSessionKey::from_session_key(
                rng,
                session_key,
                alg,
                psk,
            )?),
        }
    }

    pub fn verify(&self, s: &Signature, payload: &[u8]) -> Result<(), Error> {
        // policy check on s
        if !sig::signature_acceptable(s) {
            return Err(Error::Message(
                "Signature doesn't satisfy our policy".to_string(),
            ));
        }

        match self {
            ComponentKeyPub::Primary(pri) => Ok(s.verify(pri, payload)?),
            ComponentKeyPub::Subkey(sub) => Ok(s.verify(sub, payload)?),
        }
    }

    pub fn algorithm(&self) -> PublicKeyAlgorithm {
        match self {
            ComponentKeyPub::Primary(pri) => pri.algorithm(),
            ComponentKeyPub::Subkey(sub) => sub.algorithm(),
        }
    }

    pub fn key_id(&self) -> KeyId {
        match self {
            ComponentKeyPub::Primary(pri) => pri.key_id(),
            ComponentKeyPub::Subkey(sub) => sub.key_id(),
        }
    }

    pub fn fingerprint(&self) -> Vec<u8> {
        match self {
            ComponentKeyPub::Primary(pri) => pri.fingerprint(),
            ComponentKeyPub::Subkey(sub) => sub.fingerprint(),
        }
    }

    pub fn public_params(&self) -> &PublicParams {
        match self {
            ComponentKeyPub::Primary(pri) => pri.public_params(),
            ComponentKeyPub::Subkey(sub) => sub.public_params(),
        }
    }
}
