use crate::error::Error;
use crate::tree::Tree;
use debian_control::Control;
use std::str::FromStr;

pub trait DebianTree: Tree {
    fn get_control(&self, debian_path: &std::path::Path) -> Result<Control, Error> {
        let control_path = debian_path.join("control");

        let f = self.get_file_text(&control_path)?;

        let control = Control::from_str(std::str::from_utf8(&f).unwrap()).unwrap();

        Ok(control)
    }
}

impl DebianTree for dyn Tree {}

#[cfg(test)]
mod tests {
    use super::DebianTree;
    use crate::controldir::ControlDirFormat;
    use crate::tree::{MutableTree, Tree};

    #[test]
    fn test_open_control() {
        let td = tempfile::tempdir().unwrap();

        let wt = crate::controldir::create_standalone_workingtree(
            &td.path(),
            &ControlDirFormat::default(),
        )
        .unwrap();

        wt.mkdir(std::path::Path::new("debian")).unwrap();
        wt.put_file_bytes_non_atomic(
            std::path::Path::new("debian/control"),
            r#"Source: foo
Maintainer: Foo <blah@example.com>

"#
            .as_bytes(),
        )
        .unwrap();
        wt.add(&[std::path::Path::new("debian/control")]).unwrap();

        let control = (wt as Box<dyn crate::tree::Tree>)
            .get_control(std::path::Path::new("."))
            .unwrap();
    }
}
