/**
 * \file pappsomspp/processing/cbor/psm/evalscan/psmfeatures.cpp
 * \date 15/07/2025
 * \author Olivier Langella
 * \brief compute features on scan's PSM
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "psmfeatures.h"
#include "psmfeaturesscan.h"
#include "pappsomspp/core/utils.h"

namespace pappso
{
namespace cbor
{
namespace psm
{

PsmFeatures::PsmFeatures(std::size_t buffer_scan_size,
                                            CborStreamWriter *cbor_output_p)
  : PsmFileScanProcessAndCopy(buffer_scan_size, cbor_output_p, "psm features")
{
  m_minimumMz = 150;
  m_tandemSpectrumProcess.setMinimumMz(m_minimumMz);
  m_tandemSpectrumProcess.setNmostIntense(100);
  m_tandemSpectrumProcess.setDynamicRange(100);


  m_ionList.clear();
  m_ionList.push_back(Enums::PeptideIon::y);
  m_ionList.push_back(Enums::PeptideIon::b);

  m_fragmentTolerance = pappso::PrecisionFactory::getDaltonInstance(0.02);

  mpa_psmFeatures = new pappso::PsmFeatures(m_fragmentTolerance, m_minimumMz);
}

PsmFeatures::PsmFeatures(std::size_t buffer_scan_size,
                                            pappso::cbor::CborStreamWriter *cbor_output_p,
                                            const QJsonObject &parameters)
  : PsmFileScanProcessAndCopy(buffer_scan_size, cbor_output_p, "psm features")
{

  m_ionList.clear();
  m_ionList.push_back(Enums::PeptideIon::y);
  m_ionList.push_back(Enums::PeptideIon::b);

  if(parameters.empty())
    {
      m_minimumMz = 150;
      m_tandemSpectrumProcess.setMinimumMz(m_minimumMz);
      m_tandemSpectrumProcess.setNmostIntense(100);
      m_tandemSpectrumProcess.setDynamicRange(100);
      m_fragmentTolerance = pappso::PrecisionFactory::getDaltonInstance(0.02);
    }
  else
    {

      m_minimumMz = parameters.value("minimum_mz").toDouble();
      m_tandemSpectrumProcess.setMinimumMz(m_minimumMz);
      m_tandemSpectrumProcess.setNmostIntense(
        parameters.value("hyperscore").toObject().value("n_most_intense").toInt());
      m_tandemSpectrumProcess.setDynamicRange(
        parameters.value("hyperscore").toObject().value("dynamic_range").toInt());


      if(parameters.value("fragment_tolerance_unit").toString() == "dalton")
        {
          m_fragmentTolerance = pappso::PrecisionFactory::getDaltonInstance(
            parameters.value("fragment_tolerance").toDouble());
        }
      else if(parameters.value("fragment_tolerance_unit").toString() == "ppm")
        {
          m_fragmentTolerance = pappso::PrecisionFactory::getPpmInstance(
            parameters.value("fragment_tolerance").toDouble());
        }
    }

  mpa_psmFeatures = new pappso::PsmFeatures(m_fragmentTolerance, m_minimumMz);
}


PsmFeatures::~PsmFeatures()
{
  delete mpa_psmFeatures;
}


void
PsmFeatures::parameterMapReady(pappso::UiMonitorInterface &monitor
                                                  [[maybe_unused]])
{
  QCborMap cbor_features_parameters;

  cbor_features_parameters.insert(QString("minimum_mz"), m_minimumMz);
  cbor_features_parameters.insert(QString("fragment_tolerance_unit"),
                                  Utils::toString(m_fragmentTolerance->unit()));
  cbor_features_parameters.insert(QString("fragment_tolerance"), m_fragmentTolerance->getNominal());
  cbor_features_parameters.insert(QString("ion_list"), "y b");

  QCborMap cbor_hyperscore;
  cbor_hyperscore.insert(QString("minimum_mz"), m_minimumMz);
  cbor_hyperscore.insert(QString("n_most_intense"), m_tandemSpectrumProcess.getNmostIntense());
  cbor_hyperscore.insert(QString("dynamic_range"), m_tandemSpectrumProcess.getDynamicRange());

  cbor_features_parameters.insert(QString("hyperscore"), cbor_hyperscore);

  m_cborParameterMap.insert(QString("features"), cbor_features_parameters);

  mp_cborOutput->append("parameter_map");
  mp_cborOutput->writeCborMap(m_cborParameterMap);
}


CborScanMapBase *
PsmFeatures::newCborScanMap()
{
  return new PsmFeaturesScan(
    *this, m_tandemSpectrumProcess, m_ionList, *mpa_psmFeatures, m_fragmentTolerance);
}

} // namespace psm
} // namespace cbor
} // namespace pappso
