/*
 * Copyright (C) 2009 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/error.hpp>

#include <cerrno>
#include <cstring>

// The following switch construct is necessary because we do not know
// which error function is included by the C++ run-time library.  (The
// GNU and XSI prototypes have different return values, and slightly
// different semantics.)
namespace {
  template <typename T>
  struct strerror_switch {
  };

  template <>
  struct strerror_switch<int>
  {
    static const char* okay(char *buf, int ret) {
      if (ret == 0) {
        return buf;
      }
      return 0;
    }
  };

  template <>
  struct strerror_switch<char *>
  {
    static const char* okay(char *, char* ret) {
      if (ret != 0) {
        return ret;
      }
      return 0;
    }
  };

  template <typename T> const char*
  strerror_check(char *buf, T ret)
  {
    return strerror_switch<T>::okay(buf, ret);
  }
}

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

std::string
POSIX::ErrorString(int code)
{
  char buf[128];
  const char *buf2 =
    strerror_check(buf, strerror_r(code, buf, sizeof(buf)));
  if (buf2) {
    return std::string(buf2);
  } else {
    static const std::string str_unknown("unknown error");
    return str_unknown;
  }
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

